#-----------------------------------------------------------------------------
# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#-----------------------------------------------------------------------------

#-----------------------------------------------------------------------------
# Name:           phlatFoil.rb
# Version:        002
# Description:    Creates aerofoil sections for sketchup and phlatStuff
# Parameters:     None
# Usage:          1.  Pick aerofoil type from file (choice remains until changed).
#                 2.  Pick two points, aerofoil is drawn between them.
#                     Only works in phlatLand :-)
# Menu Item :     Tool -> Select Aerofoil...
#                 Tool -> Aerofoil
# Context Menu:   None
# Type:           Tool
# Date:           2010-12-17 13:42
# Author:         Neil Gillies  neil@iGull.co.uk
# Needs:          Cursor files "cursor_aerofoil_begin.png" and "cursor_aerofoil_end.png" to be in the plugins folder along with "phlatFoil.rb"
#                 Certainly works on Mac OSX 10.6.5 SU 7, someone other than me will need to check on PC.
#------------------- v001 ----------------------------------------------------------
# Notes about aerofoils (airfoils:-)...
# Currently, aerofoil files need to have LF line terminations (or CRLF) with the data TAB separated viz...
# xCoord TAB yCoord CR LF (or just LF)
# There are so many variations on foil formats that it is easier sticking to this 'format'.
# The good thing about standards of course is that there are so many of them =;-)
# I don't suppose this code will fail gracefully if presented with odd file formats.
# phlatFoil currently expects to see data starting at 0,0 over the top of the foil to 1,0 then back along the bottom to 0,0.
# This can also be reversed, so 1,0 to 0,0 to 1,0. If the data for top and bottom start at the leading edge, it will fail.
# Finally, the aerofoil must [should!] have a *.foil extension.
# I am also currently working on a plugin that will export an aerofoil (or whatever shape) into *.dat format for the GMFC foamcutter.
# It's all a bit rough and ready, as this is my first ruby code - sorry - maybe things will get better :-)
#------------------- v002 ----------------------------------------------------------
# Changed startup to test for phlatFoil loaded
# Changed @scaler to suit new 1..0 foil type - this means that it will no longer work with foils supplied with v001 (or rather they will be 100x bigger :-)
# This change is in line with the new 'phoilPal' plugin that converts various aerofoils from the UIUC database to *.foil format.
# Modified release notes for v001 to read 1,0 instead of 100,0.
# Due to an internal resolution of 0.001 in SU, now takes the foil data at 0..1, scaling it by 1000, create a surface, group it, then scale it back to 100%
#-----------------------------------------------------------------------------

require 'sketchup.rb'

if( not file_loaded?("phlatFoil.rb") )
	add_separator_to_menu("Tools")
	tools_menu = UI.menu("Tools")
	item = tools_menu.add_item("Select Aerofoil...") {
		getFoil
	}
	
	tools_menu = UI.menu("Tools")
	item = tools_menu.add_item("Aerofoil") {
		defaultFoil
		Sketchup.active_model.select_tool FoilTool.new
	}
	add_separator_to_menu("Tools")
end

file_loaded("phlatFoil.rb")

#-----------------------------------------------------------------------------

def getFoil
	
	b = UI.openpanel "Open Aerofoil","","*.foil"# Note that the *.foil mask only works on PC, not Mac - shows _all_ the file types.
	f = File.open(b, 'rb')
	$aeroText = IO.readlines(b)
	f.close
	
	# Array to hold loaded data
	$foilArray=Array.new($aeroText.length)
	$foilArray.map! { Array.new(3) }
	
	for i in 0..($aeroText.length-1) do
		splitter=$aeroText[i].split(/\t/)
		$foilArray[i][0]=splitter[0].to_f
		$foilArray[i][1]=splitter[1].to_f
		$foilArray[i][2]=0.00
		#puts "x="+$foilArray[i][0].to_s+"y="+$foilArray[i][1].to_s
	end
		# New array to hold scaled data
		$foilArrayScaled=Array.new($foilArray.length)
		$foilArrayScaled.map! { Array.new(3) } # Needs a z element to draw the final aerofoil
		$foilName=File.basename(b, ".foil")
		#UI.messagebox(File.basename(b, ".foil")+" loaded for Aerofoil tool")
		Sketchup.active_model.select_tool FoilTool.new
	end
	
	#-----------------------------------------------------------------------------
	
	def defaultFoil
		# Creates a dummy array of cartoon foil data
		$foilArray=Array.new(3)
		$foilArray.map! { Array.new(3) }
		$foilArray[0][0]=0.0
		$foilArray[0][1]=0.0
		$foilArray[0][2]=0.0
		$foilArray[1][0]=0.33
		$foilArray[1][1]=0.22
		$foilArray[1][2]=0.0
		$foilArray[2][0]=1.0
		$foilArray[2][1]=0.0
		$foilArray[2][2]=0.0
		
		# New array to hold scaled data
		$foilArrayScaled=Array.new($foilArray.length)
		$foilArrayScaled.map! { Array.new(3) }
		$foilName="Default Aerofoil"
	end
	
	#-----------------------------------------------------------------------------
	
	class FoilTool
	
	def initialize
		@ip1 = nil
		@ip2 = nil
		@xdown = 0
		@ydown = 0
		
		# Cursor data
		cursor_path = Sketchup.find_support_file("cursor_aerofoil_begin.png", "Plugins/")
		if cursor_path
			@beginCursor = UI.create_cursor(cursor_path, 2, 15)
		end
		
		cursor_path = Sketchup.find_support_file("cursor_aerofoil_end.png", "Plugins/")
		if cursor_path
			@endCursor = UI.create_cursor(cursor_path, 28, 15)
		end
		UI.set_cursor(@beginCursor)
		
	end
	
	def activate
		# Two points for the aerofoil endpoints.
		@ip1 = Sketchup::InputPoint.new
		@ip2 = Sketchup::InputPoint.new
		@ip = Sketchup::InputPoint.new
		@drawn = false
		
		# VCB Label
		result = Sketchup.set_status_text "Chord", SB_VCB_LABEL
		self.reset(nil)
	end
	
	def deactivate(view)
		view.invalidate if @drawn
	end
	
	def onMouseMove(flags, x, y, view)
		if( @state == 0 )
			# Leading edge of aerofoil.
			@ip.pick view, x, y
			if( @ip != @ip1 )
				# if the point has changed from the last one we got, then
				# see if we need to display the point.  We need to display it
				# if it has a display representation or if the previous point
				# was displayed.  The invalidate method on the view is used
				# to tell the view that something has changed so that you need
				# to refresh the view.
				view.invalidate if( @ip.display? or @ip1.display? )
				@ip1.copy! @ip
				
				# Set tooltip
				view.tooltip = @ip1.tooltip
				UI.set_cursor(@beginCursor)
			end
			else
			# Trailing edge of aerofoil
			# If you pass another InputPoint on the pick method of InputPoint
			# it uses that second point to do additional inferencing such as
			# parallel to an axis.
			@ip2.pick view, x, y, @ip1
			view.tooltip = @ip2.tooltip if( @ip2.valid? )
			UI.set_cursor(@endCursor)
			view.invalidate
			
			# Update chord displayed in VCB
			if( @ip2.valid? )
				length = @ip1.position.distance(@ip2.position)
				Sketchup::set_status_text length.to_s, SB_VCB_VALUE
			end
			
			# Check to see if the mouse was moved far enough to create an aerofoil.
			# Create an aerofoil by dragging or click -> click
			if( (x-@xdown).abs > 10 || (y-@ydown).abs > 10 )
				@dragging = true
			end
		end
	end
	
	def onLButtonDown(flags, x, y, view)
		# First click leading edge, second click trailing edge.
		# Create aerofoil on second click.
		if( @state == 0 )
			@ip1.pick view, x, y
			if( @ip1.valid? )
				@state = 1
				result = Sketchup.set_status_text "Select trailing edge for "+$foilName, SB_PROMPT
				@xdown = x
				@ydown = y
				UI.set_cursor(@endCursor)
			end
			else
			# Create aerofoil on second click.
			UI.set_cursor(@beginCursor)
			if( @ip2.valid? )
				self.draw_foil(@ip1.position, @ip2.position,view)
				self.reset(view)
			end
		end
		
		# Clear inference lock (if any)
		view.lock_inference
	end
	
	def onLButtonUp(flags, x, y, view)
		# If we are dragging, create the aerofoil on the mouse up event
		if( @dragging && @ip2.valid? )
			self.draw_foil(@ip1.position, @ip2.position,view)
			self.reset(view)
		end
	end
	
	# Inference lock if key pressed.
	def onKeyDown(key, repeat, flags, view)
		if( key == CONSTRAIN_MODIFIER_KEY && repeat == 1 )
			@shift_down_time = Time.now
			
			# If inference lock, then unlock
			if( view.inference_locked? )
				# Calling lock_inference with no arguments actually unlocks!
				view.lock_inference
				elsif( @state == 0 && @ip1.valid? )
				view.lock_inference @ip1
				elsif( @state == 1 && @ip2.valid? )
				view.lock_inference @ip2, @ip1
			end
		end
	end
	
	# - - - - - - - - - - - - - - - - -
	#        unlock inference
	# - - - - - - - - - - - - - - - - -
	# If user holds down shift key for more than 0.5 second, then we
	# unlock inference on release.  Otherwise, user presses shift once to 
	# lock and a second time to unlock.
	def onKeyUp(key, repeat, flags, view)
		if( key == CONSTRAIN_MODIFIER_KEY &&
		   view.inference_locked? &&
		   (Time.now - @shift_down_time) > 0.5 )
			view.lock_inference
		end
	end
	
	# - - - - - - - - - - - - - - - - -
	#        aerofoil via VCB
	# - - - - - - - - - - - - - - - - -
	def onUserText(text, view)
		# Only accept input when state is 1 (i.e. getting the second point)
		return if not @state == 1
		return if not @ip2.valid?
		
		# Mugtrap in case it's not a number ...
		begin
			value = text.to_l
			rescue
			# Error parsing the text
			UI.beep
			puts "Can't convert #{text} to a chord"
			value = nil
			Sketchup::set_status_text "", SB_VCB_VALUE
		end
		return if !value
		
		# Compute  direction and second point
		pt1 = @ip1.position
		vec = @ip2.position - pt1
		if( vec.length == 0.0 )
			UI.beep
			return
		end
		vec.length = value
		pt2 = pt1 + vec
		
		# - - - - - - - - - - - - - - - - -
		#              draw aerofoil
		# - - - - - - - - - - - - - - - - -
		self.draw_foil(pt1, pt2, view)
		self.reset(view)
	end
	
	# The draw method is called whenever the view is refreshed.  It lets the
	# tool draw any temp geometry that it needs.
	def draw(view)
		if( @ip1.valid? )
			if( @ip1.display? )
				@ip1.draw(view)
				@drawn = true
			end
			
			if( @ip2.valid? )
				@ip2.draw(view) if( @ip2.display? )
				
				# The set_color_from_line method determines what color
				# to use to draw a line based on its direction.  For example
				# red, green or blue.
				view.set_color_from_line(@ip1, @ip2)
				self.draw_geometry(@ip1.position, @ip2.position, view)
				@drawn = true
			end
		end
	end
	# - - - - - - - - - - - - - - - - -
	# User hit escape
	# - - - - - - - - - - - - - - - - -
	def onCancel(flag, view)
		self.reset(view)
	end
	
	# The following methods are not directly called from SketchUp.  They are
	# internal methods that are used to support the other methods in this class.
	
	# - - - - - - - - - - - - - - - - -
	# Reset tool
	# - - - - - - - - - - - - - - - - -
	def reset(view)
		# This variable keeps track of which point we are currently getting
		@state = 0
		
		# Display status bar prompt
		result = Sketchup.set_status_text "Select leading edge for "+$foilName, SB_PROMPT
		UI.set_cursor(@beginCursor)
		
		# Clear InputPoints
		@ip1.clear
		@ip2.clear
		
		if( view )
			view.tooltip = nil
			view.invalidate if @drawn
		end
		
		@drawn = false
		@dragging = false
	end
	
	# - - - - - - - - - - - - - - - - -
	#              draw rubber band
	# - - - - - - - - - - - - - - - - -
	def draw_geometry(pt1, pt2, view)
		view.draw_line(pt1, pt2) # Rubberband to show position.
	end
	
	# - - - - - - - - - - - - - - - - -
	#              draw aerofoil
	# - - - - - - - - - - - - - - - - -
	def draw_foil(pt1, pt2, view)
	    if  (pt1.x)!=(pt2.x)
			@scaler=(pt2.x-pt1.x)*1000 # trying to get away from the 'duplicate points' issue
			#puts @scaler.to_s
			
			for i in 0..($foilArray.length-1) do
				$foilArrayScaled[i][0]=($foilArray[i][0]*@scaler)+pt1.x#x
				$foilArrayScaled[i][1]=($foilArray[i][1]*@scaler)+pt1.y#y
				$foilArrayScaled[i][2]=($foilArray[i][2]*@scaler)+pt1.z#z
				#puts "x="+$foilArrayScaled[i][0].to_s+"y="+$foilArrayScaled[i][1].to_s+"z="+$foilArrayScaled[i][2].to_s
			end
				# Scale it back down
				t = Geom::Transformation.scaling pt1,0.001
				theFoil = Sketchup.active_model.entities.add_group
				theFoil.entities.add_face $foilArrayScaled
				theFoil = theFoil.move! t # Do it
		end
	end
		
	end # class FoilTool

