#-----------------------------------------------------------------------------
# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#-----------------------------------------------------------------------------

#-----------------------------------------------------------------------------
# Name:           phoilPal.rb
# Version:        001
# Description:    Converts various aerofoil formats to *.foil format
# Parameters:     None
# Usage:          Open aerofoil type, save to *.foil type.
# Menu Item :     plugin PhoilPal...
# Context Menu:   None
# Type:           Plugin
# Date:           2010-12-17 13:41
# Author:         Neil Gillies  neil@iGull.co.uk
# Needs:          phoilPal.rb to be in Plugins folder
#                 Certainly works on Mac OSX 10.6.5 SU 7, someone other than me will need to check on PC.
#-----------------------------------------------------------------------------
# Notes about aerofoils (airfoils:-)...
# Currently, aerofoil files need to have LF line terminations (or CRLF) with the data TAB separated viz...
# xCoord TAB yCoord CRLF (or just LF)
# There are so many variations on foil formats that it is easier sticking to this 'format'.
# The good thing about standards of course is that there are so many of them =;-)
# There is currently an issue with SU, the internal resolution is 0.001" or 0.0254mm (25.4µm)
# this causes a problem when converting datapoints to a surface -  datapoints must not be
# within < 0.001 or SU will cast a 'duplicate points' error.
# Possible 'cure' is to scale the foil data up (in phlatFoil) by [say] 1000, create the surface, group it
# then scale down to 100% again (this doesn't create duplicate point errors for some obscure reason !)
# Foils can be found here ...
# http://www.ae.illinois.edu/m-selig/ads/coord_database.html

#------------------- v001 ----------------------------------------------------------

# Major types tested so far...

# Type A
# Arad10 *.dat, ClarkY
# Line of text
# Two numbers space separated indicating no of coords top & bottom
# CRLF blank line
# x1 to 0 to 1
# SP3 xCoord SP4 yCoord CRLF
# CRLF
# x1 to 0 to 1
# SP3 xCoord SP4 yCoord CRLF

# Type B AG04.dat, e374.dat
# Mark Drela *.dat
# x1 to 0 to 1 (actually 0.999999)
# Line of text CRLF
# SP6 xCoord SP4 yCoord CRLF

# Type C
# *.COR
# x1 to 0 to 1 (actually 0.999999)
# xCoord TAB yCoord CRLF (extra blank CRLF on end)

# etc !!

# load "phoilPal.rb"

require 'sketchup.rb'

if( not file_loaded?("phoilPal.rb") )
	add_separator_to_menu("Plugins")
	UI.menu("Plugins").add_item("PhoilPal...") {phoilPal }
end

file_loaded("phoilPal.rb")

#-----------------------------------------------------------------------------

def phoilPal
	$foilArray=[]
	$b = UI.openpanel "Open Aerofoil to Convert",""
	f = File.open($b, 'rb')
	$aeroText = IO.read($b)
	f.close
	
	fileType=""
	
	# Cleanup first then to array...
	$aeroText=cleanUp($aeroText)
	$aeroText=$aeroText.split(/\n/)
		
	$multiplier=1 # Set the scale of the saved chord - currently 0..1
		
	# Let's not assume that the filename & extension form any clues,
	# that way, we won't get disappointed !
	
	# What variation of file is it?
	if ($aeroText[0] =~/[a-zA-Z]/) # Contains text, so chances are that it's a form of dat file.
		# Next line can be either real data or a pair of numbers indicating points
		# This is a very dirty method ...
		# It's going to be >10 probably (maybe :-))
			if $aeroText[1].to_f >10
				fileType = "a"
				else
				fileType = "b"
			end
		else
			fileType = "c"
			# It's probably a *.cor file
	    end
	
	case fileType
		when "a" then typeA
		when "b" then typeB
		when "c" then typeC
	end
	
end #phoilPal

#------------------- Type A Foil ----------------------------------------------------------

def typeA
	#puts "typeA"
    # Get the number of data points for each side, they _may_ be different...
	splitter=$aeroText[1].split(/\t/)
	@noOfPointsTop=splitter[0].to_i
	@noOfPointsBottom=splitter[1].to_i
	# We no longer need the first three lines...
	$aeroText.delete_at(0)
	$aeroText.delete_at(0)
	$aeroText.delete_at(0)
	# We also don't need the last datapoint first set, blank, first datapoint second set...
	@noOfPointsTop=@noOfPointsTop-1
	$aeroText.delete_at(@noOfPointsTop)
    $aeroText.delete_at(@noOfPointsTop)
    $aeroText.delete_at(@noOfPointsTop)
	@noOfPointsBottom=@noOfPointsBottom-1
	$foilArray=Array.new(@noOfPointsTop + @noOfPointsBottom)
	$foilArray.map! { Array.new(2) }
	for i in 0..(@noOfPointsTop-1) do
		splitter=$aeroText[i].split(/\t/)
		$foilArray[i][0]=splitter[0].to_f
		$foilArray[i][1]=splitter[1].to_f
	    #puts "x= "+$foilArray[i][0].to_s+" y= "+$foilArray[i][1].to_s
	end
	# Reverse the second set of data
	@counter=@noOfPointsTop
	(@noOfPointsTop+@noOfPointsBottom-1).downto(@noOfPointsTop) do |i| 
		splitter=$aeroText[i].split(/\t/)
		$foilArray[@counter][0]=splitter[0].to_f
		$foilArray[@counter][1]=splitter[1].to_f
	    puts "x= "+$foilArray[i][0].to_s+" y= "+$foilArray[i][1].to_s
		@counter=@counter+1
	end
    offsetToZero
	saveFoil
end

#------------------- Type B Foil ----------------------------------------------------------
	
def typeB
	#puts "typeB"
	# Note - there is occasionally an issue where the first and last data are the same.
	#        This creates a pair of points in the middle of the data that are the same when reversed. 
	#        SU baulks when it attempts a face from a duplicate point array if this is the case. 
	#        As a dirty fix, I'm removing the last data point from the initial array.
	
	$aeroText.pop # Remove the last datapoint (first and last are either the same or too near for SU - within 0.001)
	
	$foilArray=Array.new($aeroText.size-1)# Less one for the foil name=60 from 62-1pop total e374 [ex]
	$foilArray.map! { Array.new(2) }
	
   for i in 0...($foilArray.size) do # We're counting from 0, so miss the last one (...)
		splitter=$aeroText[i+1].split(/\t/)# Miss the first line which contains the foil name
		$foilArray[i][0]=splitter[0].to_f
		$foilArray[i][1]=splitter[1].to_f
	    #puts "x= "+$foilArray[i][0].to_s+" y= "+$foilArray[i][1].to_s
	end
	
	if $foilArray[0][0]==$foilArray[$foilArray.size-1][0] and $foilArray[0][1]==$foilArray[$foilArray.size-1][1]
	   $foilArray[0][0]=$foilArray[0][0]+0.00001
	   #puts "same datapoints"
	end
	
	# The data in type B starts at 1,so find the centre index...
	@minIndex=99999
	@minX=99999
	for i in 0...($foilArray.size) do
	  if $foilArray[i][0]<@minX
	    @minX=$foilArray[i][0]
	    @minIndex=i
	  end
	#puts@minX.to_s+"    "+@minIndex.to_s
	
	end
	
	# Temp array to hold moved data
	$tempArray=Array.new($foilArray.size)
	$tempArray.map! { Array.new(2) }

    # Take the first set of data in reverse order...
	@counter=0
	(@minIndex).downto(0) do |i| 
	 $tempArray[@counter][0]=$foilArray[i][0]
	 $tempArray[@counter][1]=$foilArray[i][1]
	 @counter=@counter+1

	end
	
    # Take the second set of data in reverse order...
	($foilArray.size-1).downto(@minIndex+1) do |i| 
	 $tempArray[@counter][0]=$foilArray[i][0]
	 $tempArray[@counter][1]=$foilArray[i][1]
	 @counter=@counter+1

	end
	
	# Then re-assemble it ...
	# There's probably a way to do this easily in ruby :-)
	for i in 0...($foilArray.size) do
		$foilArray[i][0]= $tempArray[i][0]
        $foilArray[i][1]= $tempArray[i][1]
	end

    offsetToZero
	saveFoil
	
end

#------------------- Type C Foil ----------------------------------------------------------

def typeC
	#puts "typeC"
	# Probably a *.cor file
	$aeroText.pop # Remove the last datapoint (first and last are either the same or too near for SU - within 0.001)
	$foilArray=Array.new($aeroText.size)
	$foilArray.map! { Array.new(2)}
	
   for i in 0...($foilArray.size-1) do
		splitter=$aeroText[i].split(/\t/)
		$foilArray[i][0]=splitter[0].to_f
		$foilArray[i][1]=splitter[1].to_f
	    #puts "x= "+$foilArray[i][0].to_s+" y= "+$foilArray[i][1].to_s
	end

	# The data in type C starts at 1,so find the centre index...
	@minIndex=99999
	@minX=99999
	for i in 0...($foilArray.size-1) do
	  if $foilArray[i][0]<@minX
	    @minX=$foilArray[i][0]
	    @minIndex=i
	   end
	#puts@minX.to_s+"    "+@minIndex.to_s
	end

	# Temp array to hold moved data
	$tempArray=Array.new($foilArray.size)
	$tempArray.map! { Array.new(2) }

    # Take the first set of data in reverse order...
	@counter=0
	(@minIndex).downto(0) do |i| 
	 $tempArray[@counter][0]=$foilArray[i][0]
	 $tempArray[@counter][1]=$foilArray[i][1]
	 @counter=@counter+1
	  #puts "x= "+$foilArray[i][0].to_s+" y= "+$foilArray[i][1].to_s
	end
			
    # Take the second set of data in reverse order...
	($foilArray.size-2).downto(@minIndex+1) do |i| 
	 $tempArray[@counter][0]=$foilArray[i][0]
	 $tempArray[@counter][1]=$foilArray[i][1]
	 @counter=@counter+1
	 #puts "x= "+$foilArray[i][0].to_s+" y= "+$foilArray[i][1].to_s
	end
	
	# Then re-assemble it ...
	# There's probably a way to do this easily in ruby :-)
	for i in 0...($foilArray.size-1) do
		$foilArray[i][0]= $tempArray[i][0]
        $foilArray[i][1]= $tempArray[i][1]
		#puts "x= "+$foilArray[i][0].to_s+" y= "+$foilArray[i][1].to_s
	end

    offsetToZero
	saveFoil

end

def cleanUp(theText)
	# Initial clean up
	theText.squeeze!(" ") # Remove multiple spaces
	theText.gsub!("\n ","\n")# Remove leading spaces
	theText.gsub!(" ","\t") # Replace spaces with TABs
	return theText
end

def offsetToZero
	# This difference from zero is added to all co-ords in the foilArray.
	@minX=$foilArray[0][0]
	@minY=$foilArray[0][1]
	for i in 0...($foilArray.size-1) do
		$foilArray[i][0]=($foilArray[i][0]-@minX)*$multiplier
		$foilArray[i][1]=($foilArray[i][1]-@minY)*$multiplier
		#puts "x= "+$foilArray[i][0].to_s+" y= "+$foilArray[i][1].to_s
	end

	end

def saveFoil
		foilName=File.basename($b, ".foil")
		splitter=foilName.split(/\./)
		fileName= splitter[0]+".foil"
		$b = UI.savepanel "Save Converted Aerofoil","",fileName
		if $b != nil
			f = File.new($b, "w")
			for i in 0...($foilArray.size-1) do
				f.puts $foilArray[i][0].to_s+"\t"+$foilArray[i][1].to_s + "\r\n"
			end
			f.close
	    end
end


#-----------------------------------------------------------------------------


	
	
	
