Imports System.ComponentModel
Imports System.Collections.Generic
Public Enum Motion
    RAPID = 0
    LINE = 1
    CWARC = 2
    CCARC = 3
    HOLE_I = 4
    HOLE_R = 5
    XY_PLN = 0
    XZ_PLN = 1
    YZ_PLN = 2
    I_PLN = 0
    R_PLN = 1
End Enum

Public Enum MachineType
    LATHEDIA = 0
    LATHERAD = 1
    MILL = 2
End Enum

Public Enum Axis
    Z = 0
    Y = 1
    X = 2
End Enum

Public Enum RotaryDirection
    CW = 1
    CCW = -1
End Enum

Public Enum RotaryMotionType
    BMC = 0
    CAD = 1
End Enum

''' <summary>
''' CNC Viewer
''' </summary>
''' <remarks>
''' Copyright  MacGen Programming 2006
''' Jason Titcomb
''' www.CncEdit.com
''' </remarks>
Public Class MG_BasicViewer
    Implements IDisposable
    'public members
    Public Shared ToolLayers As New Dictionary(Of Single, clsToolLayer)
    Public Shared Siblings As New List(Of MG_BasicViewer)
    Public Shared MotionBlocks As New List(Of clsMotionRecord)
    Public Enum ManipMode
        NONE
        FENCE
        PAN
        ZOOM
        ROTATE
        SELECTION
    End Enum
    Public Event AfterViewManip(ByVal mode As ManipMode, ByVal viewRect As RectangleF)
    Public Event OnStatus(ByVal msg As String, ByVal index As Integer, ByVal max As Integer)
    Public Event OnSelection(ByVal hits As List(Of clsMotionRecord))
    Public Event MouseLocation(ByVal x As Single, ByVal y As Single)
    'private members
    Private Const INT_MAXHITS As Integer = 64
    Private Const ONE_RADIAN As Single = Math.PI * 2
    Private Const PI_S As Single = Math.PI
    Private mPixelF As Single
    Private mBlipSize As Single
    Private mSinPitch As Single
    Private mSinYaw As Single
    Private mSinRoll As Single
    Private mCosPitch As Single
    Private mCosYaw As Single
    Private mCosRoll As Single
    Private mCosRot As Single
    Private mSinRot As Single
    Private mBackStep As Boolean
    Private mCurMotion As Motion
    Private mCurColor As Color
    Private mLongside As Single = 2.0
    Private mLastPos As PointF
    Private mCurGfxRec As clsMotionRecord
    Private mGfxIndex As Integer

    Private mExtentX(1) As Single
    Private mExtentY(1) As Single

    Private mPoints As New List(Of PointF)
    Private mSelectionHitLists As New List(Of clsDisplayList)
    Private mSelectionHits As New List(Of clsMotionRecord)
    Private mDisplayLists As New List(Of clsDisplayList)
    Private mWcsDisplayLists As New List(Of clsDisplayList)
    Private mMouseDownAndMoving As Boolean
    Private mMouseDownPt As Point
    Private mLastPt As Point

    Private mMtxDraw As New Drawing2D.Matrix
    Private mMtxWCS As New Drawing2D.Matrix
    Private mMtxFeedback As New Drawing2D.Matrix
    Private mMtxGeo As New Drawing2D.Matrix
    Private mViewRect As New RectangleF
    Private mClientRect As New Rectangle
    Private mSelectionRect As New RectangleF(0, 0, 0, 0)
    Private mSelectionPixRect As New Rectangle(0, 0, 4, 4)
    Private mViewportCenter As New PointF
    Private mScaleToReal As Single = 1.0
    Private mMousePtF(2) As PointF

    Private mCurPen As New Pen(Color.Blue, 0)
    Private mWCSPen As New Pen(Color.Blue, 0)
    Private mRapidDashStyle As Single() = New Single() {0.1F, 0.1F}
    Private mAxisDashStyle As Single() = New Single() {0.05F, 0.2F}

    Private mContext As BufferedGraphicsContext
    Private mGfxBuff As BufferedGraphics
    Private mGfx As Graphics

#Region "Properties"
    Private Shared mDynamicViewManipulation As Boolean = False
    <Description("Determines if the graphics are redrawn during view manipulation."), _
    Category("Custom"), _
    DefaultValue(False)> _
    Public Property DynamicViewManipulation() As Boolean
        Get
            Return mDynamicViewManipulation
        End Get
        Set(ByVal value As Boolean)
            mDynamicViewManipulation = value
        End Set
    End Property

    Private Shared mViewManipMode As ManipMode = ManipMode.NONE
    <Description("Sets or gets the view manipulation mode"), _
    Category("Custom"), _
    DefaultValue(ManipMode.NONE)> _
    Public Property ViewManipMode() As ManipMode
        Get
            Return mViewManipMode
        End Get
        Set(ByVal value As ManipMode)
            mViewManipMode = value
            For Each Sibling As MG_BasicViewer In MG_BasicViewer.Siblings
                Select Case mViewManipMode
                    Case ManipMode.PAN
                        Sibling.Cursor = Cursors.SizeAll
                    Case ManipMode.FENCE
                        Sibling.Cursor = Cursors.Cross
                    Case ManipMode.NONE
                        Sibling.Cursor = Cursors.Default
                    Case ManipMode.ROTATE
                        Sibling.Cursor = Cursors.SizeNESW
                    Case ManipMode.SELECTION
                        Sibling.Cursor = Cursors.Hand
                    Case ManipMode.ZOOM
                        Sibling.Cursor = Cursors.SizeNS
                End Select
            Next
        End Set
    End Property

    Private Shared mAxisIndicatorScale As Single = 1.0F
    <Description("Sets or gets the scale if the axis indicator"), _
    Category("Custom"), _
    DefaultValue(1.0F)> _
    Public Property AxisIndicatorScale() As Single
        Get
            Return mAxisIndicatorScale
        End Get
        Set(ByVal value As Single)
            mAxisIndicatorScale = value
        End Set
    End Property

    Private Shared mDrawAxisLines As Boolean = True
    <Description("Draw axis lines"), _
    Category("Custom"), _
    DefaultValue(True)> _
    Public Property DrawAxisLines() As Boolean
        Get
            Return mDrawAxisLines
        End Get
        Set(ByVal value As Boolean)
            mDrawAxisLines = value
        End Set
    End Property

    Private Shared mDrawAxisIndicator As Boolean = True
    <Description("Draw wcs XYZ indicator"), _
    Category("Custom"), _
    DefaultValue(True)> _
    Public Property DrawAxisIndicator() As Boolean
        Get
            Return mDrawAxisIndicator
        End Get
        Set(ByVal value As Boolean)
            mDrawAxisIndicator = value
        End Set
    End Property

    Private Shared mDrawRapidLines As Boolean = True
    <Description("Draw raid tool motion lines"), _
    Category("Custom"), _
    DefaultValue(True)> _
    Public Property DrawRapidLines() As Boolean
        Get
            Return mDrawRapidLines
        End Get
        Set(ByVal value As Boolean)
            mDrawRapidLines = value
        End Set
    End Property

    Private Shared mDrawRapidPoints As Boolean = True
    <Description("Draw raid tool motion points"), _
    Category("Custom"), _
    DefaultValue(True)> _
     Public Property DrawRapidPoints() As Boolean
        Get
            Return mDrawRapidPoints
        End Get
        Set(ByVal value As Boolean)
            mDrawRapidPoints = value
        End Set
    End Property

    Private Shared mArcAxis As Axis = Axis.Z
    <Description("Sets or gets the plane that arcs will be drawn on"), _
    Category("Custom"), _
    DefaultValue(Axis.Z)> _
    Public Property ArcAxis() As Axis
        Get
            Return mArcAxis
        End Get
        Set(ByVal value As Axis)
            mArcAxis = value
        End Set
    End Property
    Private Shared mRotaryType As RotaryMotionType = RotaryMotionType.CAD
    <Description("Sets or gets the way that fourth axis motion is interpreted"), _
    Category("Custom"), _
    DefaultValue(RotaryMotionType.CAD)> _
    Public Property RotaryType() As RotaryMotionType
        Get
            Return mRotaryType
        End Get
        Set(ByVal value As RotaryMotionType)
            mRotaryType = value
        End Set
    End Property

    Private Shared mRotaryPlane As Axis = Axis.X
    <Description("Sets or gets the plane that the fourth axis rotates on"), _
    Category("Custom"), _
    DefaultValue(Axis.X)> _
    Public Property RotaryPlane() As Axis
        Get
            Return mRotaryPlane
        End Get
        Set(ByVal value As Axis)
            mRotaryPlane = value
        End Set
    End Property

    Private Shared mRotaryDirection As RotaryDirection = MacGen.RotaryDirection.CW
    <Description("Sets or gets the direction of the fourth axis"), _
    Category("Custom"), _
    DefaultValue(MacGen.RotaryDirection.CW)> _
    Public Property RotaryDirection() As RotaryDirection
        Get
            Return mRotaryDirection
        End Get
        Set(ByVal value As RotaryDirection)
            mRotaryDirection = value
        End Set
    End Property

    Private mPitch As Single = 0
    <Description("Sets or gets the X axis rotation"), _
    Category("Custom"), _
    DefaultValue(0)> _
    Public Property Pitch() As Single
        Get
            Pitch = mPitch * (180 / PI_S)
        End Get
        Set(ByVal Value As Single)
            mPitch = Value * (PI_S / 180)
            CalcAngle()
        End Set
    End Property

    Private mRoll As Single = 0
    <Description("Sets or gets the Y axis rotation"), _
    Category("Custom"), _
    DefaultValue(0)> _
    Public Property Roll() As Single
        Get
            Roll = mRoll * (180 / PI_S)
        End Get
        Set(ByVal Value As Single)
            mRoll = Value * (PI_S / 180)
            CalcAngle()
        End Set
    End Property

    Private mYaw As Single = 0
    <Description("Sets or gets the Z axis rotation"), _
    Category("Custom"), _
    DefaultValue(0)> _
    Public Property Yaw() As Single
        Get
            Yaw = mYaw * (180 / PI_S)
        End Get
        Set(ByVal Value As Single)
            mYaw = Value * (PI_S / 180)
            CalcAngle()
        End Set
    End Property

    Private mRotary As Single = 0
    <Description("Sets or gets the fourth axis position"), _
    Category("Custom"), _
    DefaultValue(0)> _
    Public Property FourthAxis() As Single
        Set(ByVal Value As Single)
            mRotary = Value * (-RotaryDirection)
            CalcAngle()
        End Set
        Get
            Return mRotary
        End Get
    End Property

    Private mSegAngle As Single = ONE_RADIAN / 16 'angle of circular segments
    <Description("Sets the quality of arcs. >=16 AND <=720"), _
    Category("Custom"), _
    DefaultValue(16)> _
    Public WriteOnly Property ArcSegmentCount() As Integer
        Set(ByVal value As Integer)
            'Set min and max values
            If value < 16 Then value = 16
            If value > 720 Then value = 720
            mSegAngle = ONE_RADIAN / value
        End Set
    End Property

    Private Shared mBreakPoint As Integer
    <Browsable(False)> _
    Public Property BreakPoint() As Integer
        Set(ByVal value As Integer)
            If value = 0 Then
                mBreakPoint = MotionBlocks.Count - 1
            Else
                If value > MotionBlocks.Count Then
                    mBreakPoint = MotionBlocks.Count - 1
                Else
                    mBreakPoint = value
                End If
            End If
        End Set
        Get
            Return mBreakPoint
        End Get
    End Property

#End Region
    Private Sub CalcAngle()
        mCosRot = CSng(System.Math.Cos(mRotary))
        mSinRot = CSng(System.Math.Sin(mRotary))
        mSinYaw = CSng(System.Math.Sin(mYaw))
        mCosYaw = CSng(System.Math.Cos(mYaw))
        mSinRoll = CSng(System.Math.Sin(mRoll))
        mCosRoll = CSng(System.Math.Cos(mRoll))
        mSinPitch = CSng(System.Math.Sin(mPitch))
        mCosPitch = CSng(System.Math.Cos(mPitch))
    End Sub

    Private Sub MG_BasicViewer_MouseWheel(ByVal sender As Object, ByVal e As System.Windows.Forms.MouseEventArgs) Handles Me.MouseWheel
        If Math.Sign(e.Delta) = -1 Then
            ZoomScene(1.1)
        Else
            ZoomScene(0.9)
        End If
        CreateDisplayListsAndDraw()
    End Sub

    Private Sub MG_BasicViewer_MouseDown(ByVal sender As Object, ByVal e As System.Windows.Forms.MouseEventArgs) Handles Me.MouseDown
        ' Make a note that we "have the mouse".
        mMouseDownAndMoving = True
        Select Case ViewManipMode
            Case ManipMode.FENCE, ManipMode.ROTATE
                'ClearDisplayList()
            Case ManipMode.SELECTION
                If mSelectionHits.Count > 0 Then
                    RaiseEvent OnSelection(mSelectionHits)
                End If
        End Select
        ' Reset last.
        mLastPt.X = -1
        mLastPt.Y = -1
        ' Store the "starting point" for this rubber-band rectangle.
        mMouseDownPt.X = e.X
        mMouseDownPt.Y = e.Y
    End Sub

    Private Sub MG_BasicViewer_MouseMove(ByVal sender As Object, ByVal e As System.Windows.Forms.MouseEventArgs) Handles Me.MouseMove
        Static Xold As Single
        Static Yold As Single
        Dim ptCurrent As Point
        ptCurrent.X = e.X
        ptCurrent.Y = e.Y
        'Set the real coordinates of the mouse.
        'When the button whent down
        mMousePtF(0).X = mMouseDownPt.X
        mMousePtF(0).Y = mMouseDownPt.Y
        'The current location
        mMousePtF(1).X = e.X
        mMousePtF(1).Y = e.Y
        'The last location
        mMousePtF(2).X = Xold
        mMousePtF(2).Y = Yold
        'Transform the points
        mMtxFeedback.TransformPoints(mMousePtF)
        RaiseEvent MouseLocation(mMousePtF(1).X, mMousePtF(1).Y)

        Select Case ViewManipMode
            Case ManipMode.FENCE
                If (mMouseDownAndMoving) Then
                    ' Erase.
                    If (mLastPt.X <> -1) Then
                        DrawWinMouseRect(mMouseDownPt, mLastPt)
                    End If
                    ' Draw new rectangle.
                    DrawWinMouseRect(mMouseDownPt, ptCurrent)
                End If
            Case ManipMode.PAN
                If (mMouseDownAndMoving) Then
                    If mDynamicViewManipulation Then
                        PanScene((mMousePtF(1).X - mMousePtF(2).X), mMousePtF(1).Y - mMousePtF(2).Y)
                        CreateDisplayListsAndDraw()
                    Else
                        If (mLastPt.X <> -1) Then
                            DrawWinMouseLine(mMouseDownPt, mLastPt)
                        End If
                        DrawWinMouseLine(mMouseDownPt, ptCurrent)
                    End If
                End If
            Case ManipMode.ROTATE
                If mMouseDownAndMoving Then
                    Pitch += Int(-Math.Sign(Yold - e.Y))
                    Roll += Int(-Math.Sign(Xold - e.X))
                    If mDynamicViewManipulation Then
                        CreateDisplayListsAndDraw()
                    Else
                        DrawWcsOnlyToBuffer()
                    End If
                End If
            Case ManipMode.ZOOM
                If (mMouseDownAndMoving) Then
                    Dim zFact As Single
                    If e.Y > mMouseDownPt.Y Then
                        zFact = CSng(1 + ((e.Y - Yold) / Me.Height))
                    Else
                        zFact = 1 / CSng(1 + (Math.Abs(e.Y - Yold) / Me.Height))
                    End If
                    ZoomScene(zFact)
                    If mDynamicViewManipulation Then
                        CreateDisplayListsAndDraw()
                    End If
                End If
            Case ManipMode.SELECTION
                'Get a small selection viewport for selection.
                mSelectionRect.X = mMousePtF(1).X - mPixelF * mSelectionPixRect.Width / 2.0F
                mSelectionRect.Y = mMousePtF(1).Y - mPixelF * mSelectionPixRect.Height / 2.0F
                mSelectionRect.Width = mPixelF * mSelectionPixRect.Width
                mSelectionRect.Height = mPixelF * mSelectionPixRect.Height
                GetSelectionHits(mSelectionRect)
                DrawSelectionOverlay()
        End Select
        ' Update last point.
        mLastPt = ptCurrent
        Xold = e.X
        Yold = e.Y

    End Sub

    Private Sub MG_BasicViewer_MouseUp(ByVal sender As Object, ByVal e As System.Windows.Forms.MouseEventArgs) Handles Me.MouseUp
        ' Set internal flag to know we no longer "have the mouse".
        mMouseDownAndMoving = False
        ' Set flags to know that there is no "previous" line to reverse.
        mLastPt.X = -1
        mLastPt.Y = -1
        If mMouseDownPt.X = e.X Or mMouseDownPt.Y = e.Y Then Return
        Select Case ViewManipMode
            Case ManipMode.PAN
                If Not mDynamicViewManipulation Then
                    PanScene((mMousePtF(1).X - mMousePtF(0).X), mMousePtF(1).Y - mMousePtF(0).Y)
                    CreateDisplayListsAndDraw()
                End If
            Case ManipMode.ROTATE
                If Not mDynamicViewManipulation Then
                    CreateDisplayListsAndDraw()
                End If
            Case ManipMode.FENCE
                ' If we have drawn previously, draw again in that spot to remove the lines.
                If mMouseDownAndMoving And (mLastPt.X <> -1) Then
                    DrawWinMouseRect(mMouseDownPt, mLastPt)
                End If
                WindowViewport(mMousePtF(0).X, mMousePtF(0).Y, mMousePtF(1).X, mMousePtF(1).Y)
                CreateDisplayListsAndDraw()
            Case ManipMode.ZOOM
                If Not mDynamicViewManipulation Then
                    CreateDisplayListsAndDraw()
                End If
        End Select

    End Sub


    ' Convert and Normalize the points and draw the reversible frame.
    Private Sub DrawWinMouseRect(ByVal p1 As Point, ByVal p2 As Point)
        Dim rc As Rectangle
        ' Convert the points to screen coordinates.
        p1 = PointToScreen(p1)
        p2 = PointToScreen(p2)
        ' Normalize the rectangle.
        If (p1.X < p2.X) Then
            rc.X = p1.X
            rc.Width = p2.X - p1.X
        Else
            rc.X = p2.X
            rc.Width = p1.X - p2.X
        End If
        If (p1.Y < p2.Y) Then
            rc.Y = p1.Y
            rc.Height = p2.Y - p1.Y
        Else
            rc.Y = p2.Y
            rc.Height = p1.Y - p2.Y
        End If
        ' Draw the reversible frame.
        ControlPaint.DrawReversibleFrame(rc, Color.White, FrameStyle.Dashed)
    End Sub

    Private Sub DrawWinMouseLine(ByVal p1 As Point, ByVal p2 As Point)
        ' Convert the points to screen coordinates.
        p1 = PointToScreen(p1)
        p2 = PointToScreen(p2)
        ' Draw the reversible line.
        ControlPaint.DrawReversibleLine(p1, p2, Color.White)
    End Sub

    Public Sub ZoomScene(ByVal zoomFactor As Single)
        If Math.Abs(mViewRect.Width * zoomFactor) < 0.01 Then
            Return
        End If
        If Math.Abs(mViewRect.Width * zoomFactor) > 1000 Then
            Return
        End If

        Dim newWid As Single = mViewRect.Width * zoomFactor
        Dim newHt As Single = mViewRect.Height * zoomFactor

        mViewRect.X += (mViewRect.Width - newWid) / 2
        mViewRect.Y += (mViewRect.Height - newHt) / 2
        mViewRect.Width = newWid
        mViewRect.Height = newHt
        SetViewMatrix()
    End Sub

    Private Sub PanScene(ByVal deltaX As Single, ByVal deltaY As Single)
        mViewRect.X -= deltaX
        mViewRect.Y -= deltaY
        mViewportCenter.X -= deltaX
        mViewportCenter.Y -= deltaY
        SetViewMatrix()
    End Sub

    Public Sub WindowViewport(ByVal X1 As Single, ByVal Y1 As Single, ByVal X2 As Single, ByVal Y2 As Single)
        Dim temp As Single

        If (X1 > X2) Then 'convert window from right to left
            temp = X2
            X2 = X1
            X1 = temp
        End If

        If (Y1 > Y2) Then 'convert window from bottom to top
            temp = Y2
            Y2 = Y1
            Y1 = temp
        End If

        If Math.Abs(X2 - X1) < 0.01 Then
            Return
        End If
        If Math.Abs(Y2 - Y1) > 1000 Then
            Return
        End If

        mViewRect.X = X1
        mViewRect.Y = Y1
        mViewRect.Width = X2 - X1
        mViewRect.Height = Y2 - Y1
        AdjustAspect()
    End Sub

    Private Sub SetBufferContext()
        If mGfxBuff IsNot Nothing Then
            mGfxBuff.Dispose()
            mGfxBuff = Nothing
        End If
        ' Retrieves the BufferedGraphicsContext for the 
        ' current application domain.
        mContext = BufferedGraphicsManager.Current

        ' Sets the maximum size for the primary graphics buffer
        mContext.MaximumBuffer = New Size(Me.Width + 1, Me.Height + 1)

        ' Allocates a graphics buffer the size of this control
        mGfxBuff = mContext.Allocate(CreateGraphics(), New Rectangle(0, 0, Me.Width, Me.Height))
        mGfx = mGfxBuff.Graphics

    End Sub

    ''' <summary>
    ''' Sets the matrix required to draw to the specified view
    ''' </summary>
    Private Sub SetViewMatrix()
        If Single.IsInfinity(mViewRect.Width) Or Single.IsInfinity(mViewRect.Height) Then Return
        If mViewRect.Width = 0 Or mViewRect.Height = 0 Then Return

        'The ratio between the actual size of the screen and the size of the graphics.
        mScaleToReal = (mClientRect.Width / mGfx.DpiX) / mViewRect.Width

        mMtxDraw.Reset()
        mMtxDraw.Scale(mScaleToReal, mScaleToReal)
        mMtxDraw.Translate(-mViewportCenter.X, mViewportCenter.Y)
        mMtxDraw.Translate((mViewRect.Width / 2.0F), (mViewRect.Height / 2.0F))
        mMtxDraw.Scale(1, -1) 'Flip the Y


        'The matrix for the triad is the same as the other geometry but without the scale
        mMtxWCS.Reset()
        mMtxWCS.Multiply(mMtxDraw)
        mMtxWCS.Scale(1 / mScaleToReal, 1 / mScaleToReal)

        mPixelF = ((1 / mGfx.DpiX) / mScaleToReal)
        mBlipSize = (mPixelF * 3.0F)

        SetFeedbackMatrix()
    End Sub

    ''' <summary>
    ''' Adjusts the aspect of the view to match the window aspect
    ''' </summary>
    Private Sub AdjustAspect()

        If mGfx.DpiX = 0 Then Return
        If Single.IsInfinity(mViewRect.Width) Or Single.IsInfinity(mViewRect.Height) Then Return

        mViewportCenter.X = mViewRect.X + (mViewRect.Width / 2)
        mViewportCenter.Y = mViewRect.Y + (mViewRect.Height / 2)

        'Square up the viewport
        mLongside = Math.Max(mViewRect.Width, mViewRect.Height)
        mViewRect.Width = mLongside
        mViewRect.Height = mLongside

        Dim aspectRatio As Single = CSng(mClientRect.Width / mClientRect.Height)
        'Adjust the viewport aspect to match the screen aspect
        If aspectRatio >= 1.0 Then 'Wide or square screen
            'left
            mViewRect.X = mViewportCenter.X - (mLongside * aspectRatio * 0.5F)
            'width
            mViewRect.Width = mLongside * aspectRatio
            'top
            mViewRect.Y = mViewportCenter.Y - (mLongside * 0.5F)
            'height
            mViewRect.Height = mLongside
        Else 'Tall screen
            'Left
            mViewRect.X = mViewportCenter.X - (mLongside * 0.5F)
            'width
            mViewRect.Width = mLongside
            'top
            mViewRect.Y = mViewportCenter.Y - ((mLongside / aspectRatio) * 0.5F)
            'height
            mViewRect.Height = (mLongside / aspectRatio)
        End If
        SetViewMatrix()
    End Sub

    Private Sub SetFeedbackMatrix()
        mMtxFeedback.Reset()
        mMtxFeedback.Scale(mGfx.DpiX, mGfx.DpiY)
        mMtxFeedback.Multiply(mMtxDraw)
        mMtxFeedback.Invert()
    End Sub

    Private Sub DrawListsToGraphics(ByRef g As Graphics)
        If mGfxBuff Is Nothing Then Return
        mCurPen.Width = -1
        With g
            .PixelOffsetMode = Drawing2D.PixelOffsetMode.HighQuality
            .PageUnit = GraphicsUnit.Inch
            .ResetTransform()
            .MultiplyTransform(mMtxWCS)
            'Draw the axis indicator and axis lines
            For Each p As clsDisplayList In mWcsDisplayLists
                If p.Rapid Then
                    mWCSPen.DashStyle = Drawing2D.DashStyle.Custom
                    mWCSPen.DashPattern = mAxisDashStyle
                Else
                    mWCSPen.DashStyle = Drawing2D.DashStyle.Solid
                End If
                mWCSPen.Color = p.Color
                .DrawLines(mWCSPen, p.Points)
            Next
            .ResetTransform()

            'Now draw the toolpath
            mRapidDashStyle(0) = 0.05F / mScaleToReal
            mRapidDashStyle(1) = 0.05F / mScaleToReal
            .MultiplyTransform(mMtxDraw)
            For Each p As clsDisplayList In mDisplayLists
                If Not p.InView Then
                    Continue For
                End If
                If p.Rapid Then
                    mCurPen.DashStyle = Drawing2D.DashStyle.Custom
                    mCurPen.DashPattern = mRapidDashStyle
                Else
                    mCurPen.DashStyle = Drawing2D.DashStyle.Solid
                End If
                mCurPen.Color = p.Color
                LineFixUp(p.Points)
                .DrawLines(mCurPen, p.Points)
            Next
        End With
    End Sub

    'I hate this!
    'A problem exists where if the line is extended beyond the screen and it is at a sight angle
    'then it will not display completly.
    'This seems to fix the problem without too much extra processing.
    Private Sub LineFixUp(ByRef pts() As PointF)
        If pts.Length = 2 Then
            If Math.Sqrt(((pts(0).X - pts(1).X) ^ 2) + ((pts(0).Y - pts(1).Y) ^ 2)) > Me.mViewRect.Width Then
                If Math.Abs(pts(0).X - pts(1).X) < 0.001 Then
                    pts(0).X = (pts(0).X + pts(1).X) / 2
                    pts(1).X = pts(0).X
                    Return
                End If

                If Math.Abs(pts(0).Y - pts(1).Y) < 0.001 Then
                    pts(0).Y = (pts(0).Y + pts(1).Y) / 2
                    pts(1).Y = pts(0).Y
                    Return
                End If
            End If
        End If
    End Sub

    Private Sub DrawWcsOnlyToBuffer()
        If mGfxBuff Is Nothing Then Return
        CreateWcs()
        With mGfx
            .Clear(Me.BackColor)
            .PageUnit = GraphicsUnit.Inch
            .ResetTransform()
            .MultiplyTransform(mMtxWCS)
            .PixelOffsetMode = Drawing2D.PixelOffsetMode.HighQuality
            'Draw the axis indicator and axis lines
            For Each p As clsDisplayList In mWcsDisplayLists
                If p.Rapid Then
                    mWCSPen.DashStyle = Drawing2D.DashStyle.Custom
                    mWCSPen.DashPattern = mAxisDashStyle
                Else
                    mWCSPen.DashStyle = Drawing2D.DashStyle.Solid
                End If
                mWCSPen.Color = p.Color
                .DrawLines(mWCSPen, p.Points)
            Next
        End With
        mGfxBuff.Render()
    End Sub

    Private Sub DrawSelectionOverlay()
        'Draw the buffer
        If Not (mGfxBuff Is Nothing) Then
            mGfxBuff.Render()
        End If

        'Draw the selection overlay.
        mCurPen.Width = ((1 / mGfx.DpiX) / mScaleToReal) * 4
        mCurPen.EndCap = Drawing2D.LineCap.ArrowAnchor
        With Graphics.FromHwnd(Me.Handle)
            .SmoothingMode = Drawing2D.SmoothingMode.HighQuality
            .PageUnit = GraphicsUnit.Inch
            .ResetTransform()
            .MultiplyTransform(mMtxDraw)
            For Each p As clsDisplayList In mSelectionHitLists
                mCurPen.Color = p.Color
                If p.Rapid Then
                    mCurPen.DashStyle = Drawing2D.DashStyle.Dash
                Else
                    mCurPen.DashStyle = Drawing2D.DashStyle.Solid
                End If
                .DrawLines(mCurPen, p.Points)
            Next
        End With
        mCurPen.EndCap = 0
    End Sub

    Private Sub MG_BasicViewer_Paint(ByVal sender As Object, ByVal e As System.Windows.Forms.PaintEventArgs) Handles Me.Paint
        If Not (mGfxBuff Is Nothing) Then
            mGfxBuff.Render()
        End If
    End Sub

    Public Sub Init()
        SetBufferContext()
        mClientRect = ClientRectangle

        ArcSegmentCount = 16
        WindowViewport(-2.0F, -2.0F, 2.0F, 2.0F)

        SetViewMatrix()
        DrawWcsOnlyToBuffer()
    End Sub

    Private Sub MG_BasicViewer_SizeChanged(ByVal sender As Object, ByVal e As System.EventArgs) Handles Me.SizeChanged
        Init()
    End Sub

#Region "Graphics"
    Private Sub PolyCircle(ByVal Xctr As Single, ByVal Yctr As Single, ByVal Zctr As Single, _
                    ByVal Xe As Single, ByVal Xs As Single, _
                    ByVal Ye As Single, ByVal Ys As Single, _
                    ByVal Ze As Single, ByVal Zs As Single, _
                    ByVal r As Single, ByRef Startang As Single, ByRef Endang As Single, _
                    ByVal ArcDir As Integer, ByRef Wplane As Integer)

        Dim s As Integer 'counter
        Dim sngSegments As Integer ' number of angular segments
        Dim helixSeg As Single
        Dim sngAngle As Single
        Dim sngTotalAngle As Single

        sngTotalAngle = System.Math.Abs(Startang - Endang)
        sngSegments = CInt(Int(sngTotalAngle / mSegAngle))
        'Re-calculate angle increment
        sngAngle = (ArcDir * (sngTotalAngle / sngSegments))
        LineEnd4D(Xs, Ys, Zs)
        mPoints.Clear()
        Select Case Wplane
            Case Motion.XY_PLN
                helixSeg = (Ze - Zs) / sngSegments
                For s = 1 To sngSegments
                    LineEnd4D(CSng(Xctr + (r * System.Math.Cos((s * sngAngle) + Startang))), CSng(Yctr + (r * System.Math.Sin((s * sngAngle) + Startang))), Zs + helixSeg * s)
                Next s

            Case Motion.XZ_PLN
                helixSeg = (Ye - Ys) / sngSegments
                For s = 1 To sngSegments
                    LineEnd4D(CSng(Xctr + (r * System.Math.Cos((s * sngAngle) + Startang))), Ys + helixSeg * s, CSng(Zctr + (r * System.Math.Sin((s * sngAngle) + Startang))))
                Next s

            Case Motion.YZ_PLN
                helixSeg = (Xe - Xs) / sngSegments
                For s = 1 To sngSegments
                    LineEnd4D(Xs + helixSeg * s, CSng(Yctr + (r * System.Math.Cos((s * sngAngle) + Startang))), CSng(Zctr + (r * System.Math.Sin((s * sngAngle) + Startang))))
                Next s

        End Select
        LineEnd4D(Xe, Ye, Ze)
    End Sub

    Private Sub RotaryCircle(ByVal Xe As Single, ByVal Xs As Single, ByVal Ye As Single, ByVal Ys As Single, ByVal Ze As Single, ByVal Zs As Single, ByVal Startang As Single, ByVal Endang As Single, ByVal ArcDir As Single)
        Dim s As Integer 'counter
        Dim rotSegs As Integer ' number of angular segments
        Dim axisSeg1, axisSeg3, angle, totalAngle, radFromAxis As Single

        If RotaryType = RotaryMotionType.BMC Then 'Like BMC
            If ArcDir = -1 Then
                Endang = (Endang - ONE_RADIAN)
            Else
                If ArcDir = 1 And Endang < Startang Then 'take long way
                    Endang = ONE_RADIAN + Endang
                End If
            End If
        End If
        totalAngle = (Endang - Startang)
        rotSegs = CInt(System.Math.Abs(totalAngle / mSegAngle))

        Select Case RotaryPlane
            Case Axis.X  'X

                Startang = (Startang + AngleFromPoint(Zs, Ys, False) * RotaryDirection) * RotaryDirection
                Endang = (Endang + AngleFromPoint(Zs, Ys, False) * RotaryDirection) * RotaryDirection

                'Re-calculate angle increment
                angle = (totalAngle / rotSegs) * RotaryDirection
                axisSeg1 = (Xe - Xs) / rotSegs
                axisSeg3 = (Ze - Zs) / rotSegs

                radFromAxis = VectorLength(0, Ys, Zs, 0, 0, 0)
                LineEnd3D(Xs, CSng(radFromAxis * System.Math.Sin(Startang)), CSng(radFromAxis * System.Math.Cos(Startang)))
                mPoints.Clear()
                For s = 1 To rotSegs
                    radFromAxis = VectorLength(0, Ys, Zs + (axisSeg3 * s), 0, 0, 0)
                    LineEnd3D(Xs + (axisSeg1 * s), CSng(radFromAxis * System.Math.Sin((s * angle) + Startang)), CSng(radFromAxis * System.Math.Cos((s * angle) + Startang)))
                Next s
            Case Axis.Y
                Startang = (Startang - AngleFromPoint(Zs, Xs, False) * RotaryDirection) * -RotaryDirection
                Endang = (Endang - AngleFromPoint(Zs, Xs, False) * RotaryDirection) * -RotaryDirection

                'Re-calculate angle increment
                angle = (totalAngle / rotSegs) * -RotaryDirection
                axisSeg1 = (Ye - Ys) / rotSegs
                axisSeg3 = (Ze - Zs) / rotSegs
                'Debug.Print Segments
                radFromAxis = VectorLength(Xs, 0, Zs, 0, 0, 0)
                LineEnd3D(CSng(radFromAxis * System.Math.Sin(Startang)), Ys, CSng(radFromAxis * System.Math.Cos(Startang)))
                mPoints.Clear()
                For s = 1 To rotSegs
                    radFromAxis = VectorLength(Xs, 0, Zs + (axisSeg3 * s), 0, 0, 0)
                    LineEnd3D(CSng(radFromAxis * System.Math.Sin((s * angle) + Startang)), Ys + (axisSeg1 * s), CSng(radFromAxis * System.Math.Cos((s * angle) + Startang)))
                Next s
            Case Axis.Z
                'Not implemented
        End Select
        LineEnd4D(Xe, Ye, Ze)
    End Sub

    Private Sub Line3D(ByVal Xs As Single, ByVal Ys As Single, ByVal Zs As Single, ByVal Xe As Single, ByVal Ye As Single, ByVal Ze As Single)
        Dim yawXs, yawYs, rollZs As Single
        Dim yawXe, yawYe, rollZe, temp As Single

        Select Case RotaryPlane
            Case Axis.X 'x
                'X-axis start pre-rotate
                temp = (mCosRot * Ys) - (mSinRot * Zs)
                Zs = (mSinRot * Ys) + (mCosRot * Zs)
                Ys = temp
                'end pre-rotate
                temp = (mCosRot * Ye) - (mSinRot * Ze)
                Ze = (mSinRot * Ye) + (mCosRot * Ze)
                Ye = temp
            Case Axis.Y 'y
                'Y-axis start pre-rotate
                temp = (mCosRot * Zs) - (mSinRot * Xs)
                Xs = (mCosRot * Xs) + (mSinRot * Zs)
                Zs = temp
                'end
                temp = (mCosRot * Ze) - (mSinRot * Xe)
                Xe = (mCosRot * Xe) + (mSinRot * Ze)
                Ze = temp
            Case Axis.Z 'z
                'Z-axis start pre-rotate
                temp = (mCosRot * Xs) - (mSinRot * Ys)
                Xs = (mSinRot * Xs) + (mCosRot * Ys)
                Ys = temp
                'end pre-rotate
                temp = (mCosRot * Xe) - (mSinRot * Ye)
                Xe = (mSinRot * Xe) + (mCosRot * Ye)
                Ye = temp

        End Select

        'Start
        '===========================
        'Z twist
        yawXs = (mCosYaw * Xs) - (mSinYaw * Ys)
        yawYs = (mSinYaw * Xs) + (mCosYaw * Ys)

        'Y twist
        rollZs = (mCosRoll * Zs) - (mSinRoll * yawXs)
        yawXs = (mCosRoll * yawXs) + (mSinRoll * Zs) 'New X

        'X twist
        yawYs = (mCosPitch * yawYs) - (mSinPitch * rollZs) 'New Y

        'End
        '===========================
        'Z twist
        yawXe = (mCosYaw * Xe) - (mSinYaw * Ye)
        yawYe = (mSinYaw * Xe) + (mCosYaw * Ye)
        'Y twist

        rollZe = (mCosRoll * Ze) - (mSinRoll * yawXe)
        yawXe = (mCosRoll * yawXe) + (mSinRoll * Ze) 'New X
        'X twist
        yawYe = (mCosPitch * yawYe) - (mSinPitch * rollZe) 'New Y
        Line(yawXs, yawYs, yawXe, yawYe)
    End Sub

    Public Shared Function VectorLength(ByVal X1 As Single, ByVal Y1 As Single, ByVal Z1 As Single, ByVal x2 As Single, ByVal y2 As Single, ByVal Z2 As Single) As Single
        Return CSng(System.Math.Sqrt(((X1 - x2) ^ 2) + ((Y1 - y2) ^ 2) + ((Z1 - Z2) ^ 2)))
    End Function

    Public Shared Function AngleFromPoint(ByVal x As Single, ByVal y As Single, ByVal deg As Boolean) As Single
        Dim theta As Single
        If x > 0 And y > 0 Then ' Quadrant 1
            theta = CSng(System.Math.Atan(y / x))
        ElseIf x < 0 And y > 0 Then  ' Quadrant 2
            theta = CSng(System.Math.Atan(y / x) + Math.PI)
        ElseIf x < 0 And y < 0 Then  ' Quadrant 3
            theta = CSng(System.Math.Atan(y / x) + Math.PI)
        ElseIf x > 0 And y < 0 Then  ' Quadrant 4
            theta = CSng(System.Math.Atan(y / x) + 2 * Math.PI)
        End If

        ' Exceptions for points landing on an axis
        If x > 0 And y = 0 Then '0
            theta = 0
        ElseIf x = 0 And y > 0 Then  '90
            theta = Math.PI / 2
        ElseIf x < 0 And y = 0 Then  '180
            theta = Math.PI
        ElseIf x = 0 And y < 0 Then  '270
            theta = 3 * (Math.PI / 2)
        End If

        ' if you want the angle in degrees use this conversion
        If deg Then
            theta = CSng(theta * (180 / Math.PI))
        End If
        Return theta

    End Function

    Private Sub LineEnd4D(ByVal Xe As Single, ByVal Ye As Single, ByVal Ze As Single)
        Dim yawXe, yawYe, rollZe, temp As Single

        Select Case RotaryPlane
            Case Axis.X 'x
                'X-axis start pre-rotate
                'end pre-rotate
                temp = (mCosRot * Ye) - (mSinRot * Ze)
                Ze = (mSinRot * Ye) + (mCosRot * Ze)
                Ye = temp
            Case Axis.Y 'y
                'Y-axis start pre-rotate
                'end
                temp = (mCosRot * Ze) - (mSinRot * Xe)
                Xe = (mCosRot * Xe) + (mSinRot * Ze)
                Ze = temp
            Case Axis.Z 'z
                'Z-axis start pre-rotate
                'end pre-rotate
                temp = (mCosRot * Xe) - (mSinRot * Ye)
                Xe = (mSinRot * Xe) + (mCosRot * Ye)
                Ye = temp
        End Select

        'End
        '===========================
        'Z twist
        yawXe = (mCosYaw * Xe) - (mSinYaw * Ye)
        yawYe = (mSinYaw * Xe) + (mCosYaw * Ye)
        'Y twist
        rollZe = (mCosRoll * Ze) - (mSinRoll * yawXe)
        yawXe = (mCosRoll * yawXe) + (mSinRoll * Ze) 'New X
        'X twist
        yawYe = (mCosPitch * yawYe) - (mSinPitch * rollZe) 'New Y
        LineEnd(yawXe, yawYe)

    End Sub

    Private Sub LineEnd3D(ByVal Xe As Single, ByVal Ye As Single, ByVal Ze As Single)
        Dim yawXe, yawYe, rollZe As Single
        'End
        '===========================
        'Z twist
        yawXe = (mCosYaw * Xe) - (mSinYaw * Ye)
        yawYe = (mSinYaw * Xe) + (mCosYaw * Ye)
        'Y twist
        rollZe = (mCosRoll * Ze) - (mSinRoll * yawXe)
        yawXe = (mCosRoll * yawXe) + (mSinRoll * Ze) 'New X
        'X twist
        yawYe = (mCosPitch * yawYe) - (mSinPitch * rollZe) 'New Y
        LineEnd(yawXe, yawYe)
    End Sub

    Private Sub DrawEachElmt()
        Dim xleg, yleg, zleg As Single
        Dim xdir, ydir, zdir As Integer
        Dim xleg1, yleg1, zleg1, xleg2, yleg2, zleg2 As Single

        'Create a display list using any existing points
        If MG_BasicViewer.ToolLayers.ContainsKey(mCurGfxRec.Tool) Then
            If MG_BasicViewer.ToolLayers(mCurGfxRec.Tool).Hidden Then
                LineEnd4D(mCurGfxRec.Xpos, mCurGfxRec.Ypos, mCurGfxRec.Zpos)
                mPoints.Clear()
                Return
            End If
        End If

        mCurColor = mCurGfxRec.DrawClr
        mCurMotion = mCurGfxRec.MotionType

        If mCurGfxRec.Rotate Then
            Me.FourthAxis = mCurGfxRec.NewRotaryPos
            Me.ArcSegmentCount = CInt(Int((mCurGfxRec.Zpos / mLongside) * 90))
        End If

        Select Case mCurMotion
            Case Motion.RAPID
                If DrawRapidLines Then
                    If mCurGfxRec.Rotate Then
                        RotaryCircle(mCurGfxRec.Xpos, mCurGfxRec.Xold, mCurGfxRec.Ypos, mCurGfxRec.Yold, mCurGfxRec.Zpos, mCurGfxRec.Zold, mCurGfxRec.OldRotaryPos, mCurGfxRec.NewRotaryPos, mCurGfxRec.RotaryDir)
                    Else
                        xdir = System.Math.Sign(mCurGfxRec.Xpos - mCurGfxRec.Xold)
                        ydir = System.Math.Sign(mCurGfxRec.Ypos - mCurGfxRec.Yold)
                        zdir = System.Math.Sign(mCurGfxRec.Zpos - mCurGfxRec.Zold)

                        xleg = System.Math.Abs(mCurGfxRec.Xpos - mCurGfxRec.Xold)
                        yleg = System.Math.Abs(mCurGfxRec.Ypos - mCurGfxRec.Yold)
                        zleg = System.Math.Abs(mCurGfxRec.Zpos - mCurGfxRec.Zold)

                        If xleg <= yleg And yleg <= zleg Then
                            xleg1 = mCurGfxRec.Xpos
                            yleg1 = mCurGfxRec.Yold + xleg * ydir
                            zleg1 = mCurGfxRec.Zold + xleg * zdir
                            xleg2 = mCurGfxRec.Xpos
                            yleg2 = mCurGfxRec.Ypos
                            zleg2 = mCurGfxRec.Zold + yleg * zdir
                        ElseIf xleg <= zleg And zleg <= yleg Then
                            xleg1 = mCurGfxRec.Xpos
                            yleg1 = mCurGfxRec.Yold + xleg * ydir
                            zleg1 = mCurGfxRec.Zold + xleg * zdir
                            xleg2 = mCurGfxRec.Xpos
                            yleg2 = mCurGfxRec.Yold + zleg * ydir
                            zleg2 = mCurGfxRec.Zpos
                        ElseIf zleg <= yleg And yleg <= xleg Then
                            xleg1 = mCurGfxRec.Xold + zleg * xdir
                            yleg1 = mCurGfxRec.Yold + zleg * ydir
                            zleg1 = mCurGfxRec.Zpos
                            xleg2 = mCurGfxRec.Xold + yleg * xdir
                            yleg2 = mCurGfxRec.Ypos
                            zleg2 = mCurGfxRec.Zpos
                        ElseIf zleg <= xleg And xleg <= yleg Then
                            xleg1 = mCurGfxRec.Xold + zleg * xdir
                            yleg1 = mCurGfxRec.Yold + zleg * ydir
                            zleg1 = mCurGfxRec.Zpos
                            xleg2 = mCurGfxRec.Xpos
                            yleg2 = mCurGfxRec.Yold + xleg * ydir
                            zleg2 = mCurGfxRec.Zpos
                        ElseIf yleg <= zleg And zleg <= xleg Then
                            xleg1 = mCurGfxRec.Xold + yleg * xdir
                            yleg1 = mCurGfxRec.Ypos
                            zleg1 = mCurGfxRec.Zold + yleg * zdir
                            xleg2 = mCurGfxRec.Xold + zleg * xdir
                            yleg2 = mCurGfxRec.Ypos
                            zleg2 = mCurGfxRec.Zpos
                        ElseIf yleg <= xleg And xleg <= zleg Then
                            xleg1 = mCurGfxRec.Xold + yleg * xdir
                            yleg1 = mCurGfxRec.Ypos
                            zleg1 = mCurGfxRec.Zold + yleg * zdir
                            xleg2 = mCurGfxRec.Xpos
                            yleg2 = mCurGfxRec.Ypos
                            zleg2 = mCurGfxRec.Zold + xleg * zdir
                        End If
                        LineEnd3D(xleg1, yleg1, zleg1)
                        LineEnd4D(xleg2, yleg2, zleg2)
                        LineEnd4D(mCurGfxRec.Xpos, mCurGfxRec.Ypos, mCurGfxRec.Zpos)
                    End If
                End If
                CreateDisplayList(True) 'Draw any existing lines
                'Draw a rapid blip if required
                If DrawRapidPoints Then
                    'Set the last point
                    LineEnd4D(mCurGfxRec.Xpos, mCurGfxRec.Ypos, mCurGfxRec.Zpos)
                    mPoints.Clear()
                    'mCurMotion = Motion.LINE
                    DrawBlip(mLastPos)
                    CreateDisplayList(False)
                    mCurMotion = mCurGfxRec.MotionType
                End If

            Case Motion.HOLE_I, Motion.HOLE_R
                If DrawRapidLines Then

                    'The direction
                    xdir = System.Math.Sign(mCurGfxRec.Xpos - mCurGfxRec.Xold)
                    ydir = System.Math.Sign(mCurGfxRec.Ypos - mCurGfxRec.Yold)

                    xleg = System.Math.Abs(mCurGfxRec.Xpos - mCurGfxRec.Xold)
                    yleg = System.Math.Abs(mCurGfxRec.Ypos - mCurGfxRec.Yold)

                    'A rotary move is on the drill cycle line
                    If mCurGfxRec.Rotate Then
                        If mCurGfxRec.MotionType = Motion.HOLE_I Then 'Return to inital Z
                            RotaryCircle(mCurGfxRec.Xpos, mCurGfxRec.Xold, mCurGfxRec.Ypos, mCurGfxRec.Yold, mCurGfxRec.DrillClear, mCurGfxRec.DrillClear, mCurGfxRec.OldRotaryPos, mCurGfxRec.NewRotaryPos, mCurGfxRec.RotaryDir)

                        Else
                            RotaryCircle(mCurGfxRec.Xpos, mCurGfxRec.Xold, mCurGfxRec.Ypos, mCurGfxRec.Yold, mCurGfxRec.Rpoint, mCurGfxRec.Rpoint, mCurGfxRec.OldRotaryPos, mCurGfxRec.NewRotaryPos, mCurGfxRec.RotaryDir)
                        End If
                        LineEnd4D(mCurGfxRec.Xpos, mCurGfxRec.Ypos, mCurGfxRec.Rpoint)
                    Else

                        If xleg <= yleg Then
                            'The first end point
                            'Move in the direction of each axis by the length of the shortest axis
                            xleg1 = mCurGfxRec.Xold + xleg * xdir
                            yleg1 = mCurGfxRec.Yold + xleg * ydir
                        End If

                        If xleg >= yleg Then
                            'The first end point
                            'Move in the direction of each axis by the length of the shortest axis
                            xleg1 = mCurGfxRec.Xold + yleg * xdir
                            yleg1 = mCurGfxRec.Yold + yleg * ydir
                        End If
                        'Dog-leg hole positioning
                        If mCurGfxRec.MotionType = Motion.HOLE_I Then 'Return to inital Z
                            'Dog-leg
                            Line3D(mCurGfxRec.Xold, mCurGfxRec.Yold, mCurGfxRec.DrillClear, xleg1, yleg1, mCurGfxRec.DrillClear)
                            LineEnd4D(mCurGfxRec.Xpos, mCurGfxRec.Ypos, mCurGfxRec.DrillClear)
                            LineEnd4D(mCurGfxRec.Xpos, mCurGfxRec.Ypos, mCurGfxRec.Rpoint)
                        Else
                            Line3D(mCurGfxRec.Xold, mCurGfxRec.Yold, mCurGfxRec.Zold, mCurGfxRec.Xold, mCurGfxRec.Yold, mCurGfxRec.Rpoint)
                            LineEnd4D(xleg1, yleg1, mCurGfxRec.Rpoint)
                            LineEnd4D(mCurGfxRec.Xpos, mCurGfxRec.Ypos, mCurGfxRec.Rpoint)
                        End If
                    End If
                End If

                CreateDisplayList(True) 'Draw any existing lines
                'Draw the hole line
                Line3D(mCurGfxRec.Xpos, mCurGfxRec.Ypos, mCurGfxRec.Rpoint, mCurGfxRec.Xpos, mCurGfxRec.Ypos, mCurGfxRec.Zpos)

                If DrawRapidPoints Then 'Draw a small circle
                    CreateDisplayList(False) 'Draw any existing lines
                    ArcSegmentCount = 8
                    PolyCircle(mCurGfxRec.Xpos, mCurGfxRec.Ypos, mCurGfxRec.Zpos, _
                                                mCurGfxRec.Xpos + mBlipSize, _
                                                mCurGfxRec.Xpos + mBlipSize, _
                                                mCurGfxRec.Ypos, mCurGfxRec.Ypos, _
                                                mCurGfxRec.Zpos, mCurGfxRec.Zpos, _
                                                mBlipSize, 0, ONE_RADIAN, -1, 0)
                End If

            Case Motion.LINE
                If mCurGfxRec.Rotate Then
                    RotaryCircle(mCurGfxRec.Xpos, mCurGfxRec.Xold, mCurGfxRec.Ypos, mCurGfxRec.Yold, mCurGfxRec.Zpos, mCurGfxRec.Zold, mCurGfxRec.OldRotaryPos, mCurGfxRec.NewRotaryPos, mCurGfxRec.RotaryDir)
                Else
                    Line3D(mCurGfxRec.Xold, mCurGfxRec.Yold, mCurGfxRec.Zold, mCurGfxRec.Xpos, mCurGfxRec.Ypos, mCurGfxRec.Zpos)
                End If
            Case Motion.CCARC
                ArcSegmentCount = CInt(Int((mCurGfxRec.Rad / mLongside) * 360))
                PolyCircle(mCurGfxRec.Xcentr, mCurGfxRec.Ycentr, mCurGfxRec.Zcentr, mCurGfxRec.Xpos, mCurGfxRec.Xold, mCurGfxRec.Ypos, mCurGfxRec.Yold, mCurGfxRec.Zpos, mCurGfxRec.Zold, mCurGfxRec.Rad, mCurGfxRec.Sang, mCurGfxRec.Eang, 1, mCurGfxRec.WrkPlane)
            Case Motion.CWARC
                ArcSegmentCount = CInt(Int((mCurGfxRec.Rad / mLongside) * 360))
                PolyCircle(mCurGfxRec.Xcentr, mCurGfxRec.Ycentr, mCurGfxRec.Zcentr, mCurGfxRec.Xpos, mCurGfxRec.Xold, mCurGfxRec.Ypos, mCurGfxRec.Yold, mCurGfxRec.Zpos, mCurGfxRec.Zold, mCurGfxRec.Rad, mCurGfxRec.Sang, mCurGfxRec.Eang, -1, mCurGfxRec.WrkPlane)
        End Select
        CreateDisplayList(False)
    End Sub

    'Draw un-rotated rectangle as a rapid point indication.   
    Private Sub DrawBlip(ByVal p As PointF)
        mPoints.Clear()
        Line(p.X - mBlipSize, p.Y - mBlipSize, p.X + mBlipSize, p.Y - mBlipSize)
        LineEnd(p.X + mBlipSize, p.Y + mBlipSize)
        LineEnd(p.X - mBlipSize, p.Y + mBlipSize)
        LineEnd(p.X - mBlipSize, p.Y - mBlipSize)
    End Sub

    Public Overloads Sub Redraw(ByVal allSiblings As Boolean)
        If allSiblings Then
            For Each sib As MG_BasicViewer In MG_BasicViewer.Siblings
                If sib.ParentForm.Name = Me.ParentForm.Name Then
                    sib.CreateDisplayListsAndDraw()
                End If
            Next
        Else
            CreateDisplayListsAndDraw()
        End If
    End Sub

    Public Sub FindExtents()
        If Not Visible Then Return
        If MotionBlocks.Count = 0 Then Return

        mExtentX(0) = Single.MaxValue
        mExtentX(1) = Single.MinValue
        mExtentY(0) = Single.MaxValue
        mExtentY(1) = Single.MinValue

        Dim drawRapidPointsStatus As Boolean = DrawRapidPoints
        DrawRapidPoints = False 'Disable rapid points for speed
        CreateDisplayLists()
        DrawRapidPoints = drawRapidPointsStatus
        If MotionBlocks.Count > 0 Then
            For Each l As clsDisplayList In mDisplayLists
                For Each p As PointF In l.Points
                    mExtentX(0) = Math.Min(mExtentX(0), p.X)
                    mExtentX(1) = Math.Max(mExtentX(1), p.X)
                    mExtentY(0) = Math.Min(mExtentY(0), p.Y)
                    mExtentY(1) = Math.Max(mExtentY(1), p.Y)
                Next
            Next
        Else
            mExtentX(0) = -1.0F
            mExtentX(1) = 1.0F
            mExtentY(0) = -1.0F
            mExtentY(1) = 1.0F
        End If

        mViewRect.X = mExtentX(0)
        mViewRect.Width = mExtentX(1) - mExtentX(0)
        mViewRect.Y = mExtentY(0)
        mViewRect.Height = mExtentY(1) - mExtentY(0)
        If Single.IsNegativeInfinity(mViewRect.Width) Then Return
        If Single.IsNegativeInfinity(mViewRect.Height) Then Return
        mViewRect.Inflate(mViewRect.Width * 0.01F, mViewRect.Height * 0.01F)

        AdjustAspect()
        CreateDisplayListsAndDraw()
    End Sub

    Private Sub CreateDisplayLists()
        mDisplayLists.Clear()
        mPoints.Clear()

        mLastPos.X = 0
        mLastPos.Y = 0
        For mGfxIndex = 0 To mBreakPoint
            mCurGfxRec = MotionBlocks(mGfxIndex)
            DrawEachElmt() 'Draws geometry
        Next
    End Sub

    Private Sub DrawDisplayLists()
        CreateWcs()
        SetInViewStatus(Me.mViewRect)
        mGfx.Clear(Me.BackColor)
        DrawListsToGraphics(mGfx)
        mGfxBuff.Render()
    End Sub

    Private Sub CreateDisplayListsAndDraw()
        CreateDisplayLists()
        DrawDisplayLists()
    End Sub

    Private Sub CreateWcs()
        If Not Visible Then Return

        mWcsDisplayLists.Clear()
        mPoints.Clear()
        FourthAxis = 0
        If DrawAxisLines Then
            mCurMotion = Motion.RAPID
            'Y axis line
            Line3D(0, 0, 0, 0, -mLongside * 10, 0)
            Me.CreateWcsPath(Color.Gray)
            Line3D(0, 0, 0, 0, mLongside * 10, 0)
            Me.CreateWcsPath(Color.Gray)

            'X axis line
            Line3D(0, 0, 0, mLongside * 10, 0, 0)
            Me.CreateWcsPath(Color.Gray)
            Line3D(0, 0, 0, -mLongside * 10, 0, 0)
            Me.CreateWcsPath(Color.Gray)

            'Z Axis line
            Line3D(0, 0, 0, 0, 0, mLongside * 10)
            Me.CreateWcsPath(Color.Gray)
            Line3D(0, 0, 0, 0, 0, -mLongside * 10)
            Me.CreateWcsPath(Color.Gray)
        End If

        If DrawAxisIndicator Then
            'Axis indicators
            mCurMotion = Motion.LINE
            'X indicator
            Line3D(0, 0, 0, 1, 0, 0)
            Line3D(1, 0, 0, 0.9, 0.05, 0)
            Line3D(1, 0, 0, 0.9, -0.05, 0)
            CreateWcsPath(Color.DarkKhaki)
            'Draw the letter X
            Line3D(0.7, 0.1, 0, 0.9, 0.4, 0)
            CreateWcsPath(Color.DarkKhaki)
            Line3D(0.9, 0.1, 0, 0.7, 0.4, 0)
            CreateWcsPath(Color.DarkKhaki)

            'Y indicator
            Line3D(0, 0, 0, 0, 1, 0)
            CreateWcsPath(Color.DarkGreen)
            Line3D(0, 1, 0, -0.05, 0.9, 0)
            CreateWcsPath(Color.DarkGreen)
            Line3D(0, 1, 0, 0.05, 0.9, 0)
            CreateWcsPath(Color.DarkGreen)
            'Draw the letter Y
            Line3D(-0.2, 0.7, 0, -0.2, 0.85, 0)
            CreateWcsPath(Color.DarkGreen)
            Line3D(-0.2, 0.85, 0, -0.3, 1, 0)
            CreateWcsPath(Color.DarkGreen)
            Line3D(-0.2, 0.85, 0, -0.1, 1, 0)
            CreateWcsPath(Color.DarkGreen)

            'Z indicator
            Line3D(0, 0, 0, 0, 0, 1)
            Line3D(0, 0, 1, 0.1, 0, 0.8)
            CreateWcsPath(Color.DarkRed)

            PolyCircle(0, 0, 0, 0.1, 0.1, 0, 0, 0.8, 0.8, 0.1, 0, ONE_RADIAN, 1, Axis.Z)
            CreateWcsPath(Color.DarkRed)

            'Draw the letter Z
            Line3D(-0.2, 0, 0.7, -0.4, 0, 0.7)
            Line3D(-0.2, 0, 0.95, -0.4, 0, 0.95)
            Line3D(-0.2, 0, 0.95, -0.4, 0, 0.7)
            CreateWcsPath(Color.DarkRed)
        End If
    End Sub

    Public Sub GatherTools()
        Dim lastTool As Single
        ToolLayers.Clear()
        For Each blk As clsMotionRecord In MotionBlocks
            If lastTool <> blk.Tool Then
                lastTool = blk.Tool
                If Not ToolLayers.ContainsKey(blk.Tool) Then
                    ToolLayers.Add(blk.Tool, New clsToolLayer(blk.Tool, blk.DrawClr))
                End If
            End If
        Next
    End Sub

    'Returns the number of hits inside the referenced rectangle.
    Private Sub GetSelectionHits(ByVal rect As RectangleF)
        Dim maxHits As Integer = 0
        Dim cadRect As New clsCadRect(rect.X, rect.Y, rect.Width, rect.Height)
        mSelectionHits.Clear()
        mSelectionHitLists.Clear()
        If MotionBlocks.Count > 0 Then
            For Each l As clsDisplayList In mDisplayLists
                If l.InView Then
                    'Iterate in sets of 2 
                    For r As Integer = 0 To l.Points.Length - 2
                        If maxHits >= INT_MAXHITS Then Return
                        If cadRect.IntersectsLine(l.Points(r), l.Points(r + 1)) Then
                            mSelectionHits.Add(MotionBlocks(l.ParentIndex))
                            mSelectionHitLists.Add(l)
                            maxHits += 1
                            Exit For
                        End If
                    Next
                End If
            Next
        End If
    End Sub

    Private Sub SetInViewStatus(ByVal rect As RectangleF)
        Dim cadRect As New clsCadRect(rect.X, rect.Y, rect.Width, rect.Height)
        For Each l As clsDisplayList In mDisplayLists
            'Iterate in sets of 2 
            For r As Integer = 0 To l.Points.Length - 2
                l.InView = False
                If cadRect.IntersectsLine(l.Points(r), l.Points(r + 1)) Then
                    l.InView = True
                    Exit For
                End If
            Next
        Next
    End Sub


    Private Sub ClearDisplayList()
        mDisplayLists.Clear()
        mPoints.Clear()
    End Sub

    Private Sub CreateDisplayList(ByVal rapid As Boolean)
        Dim p As New clsDisplayList
        If (mPoints.Count < 2) Then Return
        With p
            .Color = mCurColor
            .Rapid = rapid
            .ParentIndex = mGfxIndex
            .Points = mPoints.ToArray
        End With
        mDisplayLists.Add(p)
        mPoints.Clear()
    End Sub

    'Axis lines
    Private Sub CreateWcsPath(ByVal clr As Color)
        Dim p As New clsDisplayList
        If (mPoints.Count < 2) Then Return
        With p
            .Color = clr
            .Rapid = (mCurMotion = Motion.RAPID)
            .Points = mPoints.ToArray
        End With
        mWcsDisplayLists.Add(p)
        mPoints.Clear()
    End Sub

    Private Sub LineEnd(ByVal x2 As Single, ByVal y2 As Single)
        If mLastPos.X <> x2 And mLastPos.Y <> y2 Then
            mPoints.Add(mLastPos)
        End If
        mPoints.Add(New PointF(x2, y2))
        mLastPos.X = x2
        mLastPos.Y = y2
    End Sub

    Private Sub Line(ByVal x1 As Single, ByVal y1 As Single, ByVal x2 As Single, ByVal y2 As Single)
        mPoints.Add(New PointF(x1, y1))
        mPoints.Add(New PointF(x2, y2))
        mLastPos.X = x2
        mLastPos.Y = y2
    End Sub

#End Region

    Public Sub New()

        ' This call is required by the Windows Form Designer.
        InitializeComponent()

        ' Add any initialization after the InitializeComponent() call.
        Siblings.Add(Me)
        SetStyle(ControlStyles.AllPaintingInWmPaint Or ControlStyles.UserPaint, True)
        SetStyle(ControlStyles.OptimizedDoubleBuffer, False)
        ' Retrieves the BufferedGraphicsContext for the current application domain.
        mContext = BufferedGraphicsManager.Current
    End Sub

    Protected Overrides Sub Finalize()
        If mGfxBuff IsNot Nothing Then
            mGfxBuff.Dispose()
        End If
        If mMtxDraw IsNot Nothing Then
            mMtxDraw.Dispose()
        End If
        If mCurPen IsNot Nothing Then
            mCurPen.Dispose()
        End If
        If mWCSPen IsNot Nothing Then
            mWCSPen.Dispose()
        End If
        MyBase.Finalize()
    End Sub

    Protected Overrides Sub Dispose(ByVal disposing As Boolean)
        Siblings.Remove(Me)
        If disposing AndAlso components IsNot Nothing Then
            components.Dispose()
        End If
        MyBase.Dispose(disposing)
        System.GC.SuppressFinalize(Me)
    End Sub

    Private Sub MG_BasicViewer_VisibleChanged(ByVal sender As Object, ByVal e As System.EventArgs) Handles Me.VisibleChanged
        If Me.Visible = False Then
            'Reclaim a little memory
            Me.mDisplayLists.Clear()
        End If
    End Sub
End Class
