require 'sketchup.rb'
require 'Phlatboyz/Constants.rb'

require 'Phlatboyz/PhlatboyzMethods.rb'
require 'Phlatboyz/PhlatOffset.rb'

require 'Phlatboyz/PhlatMill.rb'
require 'Phlatboyz/PhlatTool.rb'
require 'Phlatboyz/PhlatCut.rb'
require 'Phlatboyz/PSUpgrade.rb'

module PhlatScript

  class GcodeUtil < PhlatTool

    @@x_save = nil
    @@y_save = nil
    @@cut_depth_save = nil
    @@edge_type_save = nil
  
    def initialize
      @tooltype = 3
      @tooltip = PhlatScript.getString("Phlatboyz GCode")
      @largeIcon = "images/gcode_large.png"
      @smallIcon = "images/gcode_small.png"
      @statusText = PhlatScript.getString("Phlatboyz GCode")
      @menuItem = PhlatScript.getString("GCode")
      @menuText = PhlatScript.getString("GCode")
    end

    def select
      GcodeUtil.generate_gcode
    end

    def GcodeUtil.generate_gcode
      if PSUpgrader.upgrade
        UI.messagebox("GCode generation has been aborted due to the upgrade")
        return
      end
      model = Sketchup.active_model
      if(enter_file_dialog(model))
        # first get the material thickness from the model dictionary
        material_thickness = Sketchup.active_model.get_attribute $dict_name, $dict_material_thickness, $default_material_thickness 
        if(material_thickness)

          begin
            output_directory_name = model.get_attribute $dict_name, $dict_output_directory_name, $default_directory_name
            output_file_name = model.get_attribute $dict_name, $dict_output_file_name, $default_file_name
            current_bit_diameter = model.get_attribute $dict_name, $dict_bit_diameter, $default_bit_diameter

            # TODO check for existing / on the end of output_directory_name
            absolute_File_name = output_directory_name + output_file_name
            
            safe_array = get_safe_array()
            min_x = 0.0
            min_y = 0.0
            max_x = safe_array[2]
            max_y = safe_array[3]
            safe_area_points = get_safe_area_point3d_array()

            min_max_array = [min_x, max_x, min_y, max_y, $min_z, $max_z]
            #aMill = CNCMill.new(nil, nil, absolute_File_name, min_max_array)
            aMill = PhlatMill.new(absolute_File_name, min_max_array)
            
            aMill.set_bit_diam(current_bit_diameter)
            
            #puts("starting aMill absolute_File_name="+absolute_File_name)
            aMill.job_start()

            loop_root = LoopNodeFromEntities(Sketchup.active_model.active_entities, aMill, material_thickness)
            loop_root.sort
            millLoopNode(aMill, loop_root, material_thickness)

            #puts("done milling")
            
            aMill.home()
            # retracts the milling head and
            # and then moves it home.  This
            # prevents accidental milling 
            # through your work piece when 
            # moving home.

            #puts("finishing up")
            aMill.job_finish() # output housekeeping code
          rescue
            UI.messagebox "GcodeUtil.generate_gcode failed; Error:"+$!
          end
        else
          UI.messagebox($phlatboyzStrings.GetString("You must define the material thickness."))
        end
      end
    end
    
    private

    def GcodeUtil.LoopNodeFromEntities(entities, aMill, material_thickness)
      model = Sketchup.active_model
            safe_area_points = get_safe_area_point3d_array()
            # find all outside loops
            loops = []
            groups = []
            phlatcuts = []
            entities.each { |e|
              if e.kind_of?(Sketchup::Face)
                has_edges = false
                # only keep loops that contain phlatcuts
                e.outer_loop.edges.each { |edge|
                  pc = PhlatCut.from_edge(edge)
                  has_edges = ((!pc.nil?) && (pc.in_polygon?(safe_area_points)))
                  break if has_edges
                }
                loops.push(LoopNode.new(e.outer_loop)) if has_edges
              elsif e.kind_of?(Sketchup::Edge)
                # make sure that all edges are marked as not processed
                pc = PhlatCut.from_edge(e)
                if (pc) 
                  pc.processed = (false) 
                  phlatcuts.push(pc) if ((pc.in_polygon?(safe_area_points)) && ((pc.kind_of? PhlatScript::PlungeCut) || (pc.kind_of? PhlatScript::CenterLineCut)))
                end
              elsif e.kind_of?(Sketchup::Group)
                groups.push(e)
              end
            }

            groups.each { |e|  
                # this is a bit hacky and we should try to use a transformation based on 
                # the group.local_bounds.corner(0) in the future
                group_name = e.name
                aMill.cncPrint("(Group: #{group_name})\n") if !group_name.empty?
          			model.start_operation "Exploding Group", true
                es = e.explode
                gnode = LoopNodeFromEntities(es, aMill, material_thickness)
                gnode.sort
                millLoopNode(aMill, gnode, material_thickness)
                # abort the group explode
                model.abort_operation
                aMill.cncPrint("(Group complete: #{group_name})\n") if !group_name.empty?
            }
            loops.flatten!
            loops.uniq!
            puts("Located #{loops.length.to_s} loops containing PhlatCuts")

            # nest the loops so that loops with all vertices contained in another loop can be cut first
            loop_root = LoopNode.new(nil)
            0.upto(loops.length-1) { |i|
              found_parent = false
              0.upto(loops.length-1) { |x|
                next if (i==x) || (found_parent)
                if (!found_parent) && (Geom.point_in_polygon_2D(loops[i].loop.vertices.first, loops[x].loop.vertices, false))
                  found_parent = true
                  loops[x].children.push(loops[i])
                end
              }
              if !found_parent
                loop_root.children.push(loops[i])
              end
            }

            # push all the plunge, centerline and fold cuts into the proper loop node
            phlatcuts.each { |pc| 
              loop_root.find_container(pc)
              pc.processed = true
            }
      return loop_root
    end

    def GcodeUtil.millLoopNode(aMill, loopNode, material_thickness)
      # always mill the child loops first
      loopNode.children.each{ |childloop|
        millLoopNode(aMill, childloop, material_thickness)
      }
      millEdges(aMill, loopNode.sorted_cuts, material_thickness)

      # finally we can walk the loop and make it's cuts
      edges = []
      reverse = false
      pe = nil
      if !loopNode.loop.nil?
        loopNode.loop.edgeuses.each{ |eu|
          pe = PhlatCut.from_edge(eu.edge)
          if (pe) && (!pe.processed)
            reverse = reverse || (pe.kind_of?(PhlatScript::InsideCut))
            edges.push(pe)
            pe.processed = true
          end
        }
        loopNode.loop_start.downto(0) { |x|
          edges.push(edges.shift) if x > 0
        }
        edges.reverse! if reverse
      end 
      edges.compact!
      millEdges(aMill, edges, material_thickness, reverse)
    end

    def GcodeUtil.millEdges(aMill, edges, material_thickness, reverse=false)
      if (edges) && (!edges.empty?)
        begin
          mirror = get_safe_reflection_translation()
          trans = get_safe_origin_translation()
          trans = trans * mirror if $reflection_output
        
          @@x_save = nil
          @@y_save = nil
          @@cut_depth_save = nil
          @@edge_type_save = nil
          
          aMill.retract()

          save_point = nil
          edges.each do | wrapped_edge |
            save_point = millFlatEdge(aMill, wrapped_edge, trans, save_point, material_thickness, reverse)
          end
        rescue
          UI.messagebox "Exception in millEdges "+$!
        end
      end
    end

    def GcodeUtil.millFlatEdge(aMill, phlatcut, in_trans, in_save_point, in_material_thickness, reverse=false)
      returnPoint = nil
      begin
        if(in_save_point == nil)
          @@x_save = nil
          @@y_save = nil
          @@cut_depth_save = nil
        else
          @@x_save = in_save_point.x
          @@y_save = in_save_point.y
          @@cut_depth_save = in_save_point.z
        end

        cut_started = false
        point = nil
        cut_depth = 0

        phlatcut.cut_points(reverse) { |cp, cut_factor|
          cut_depth = -1.0 * in_material_thickness * (cut_factor.to_f/100).to_f
          # transform the point if a transformation is provided
          point = (in_trans ? (cp.transform(in_trans)) : cp)
          # retract if this cut does not start where the last one ended
          if ((@@x_save != point.x) || (@@y_save != point.y) || (@@cut_depth_save != cut_depth))
            if (!cut_started)
              aMill.retract()
              aMill.move(point.x, point.y)
              aMill.plung(cut_depth)
            else
              if ((phlatcut.kind_of? PhlatArc) && (phlatcut.is_arc?) && ((@@x_save != point.x) || (@@y_save != point.y)))
                g3 = reverse ? !phlatcut.g3? : phlatcut.g3?
                aMill.arcmove(point.x, point.y, phlatcut.radius, g3, cut_depth)
              else
                aMill.move(point.x, point.y, cut_depth)
              end
            end
          end
          cut_started = true
          @@x_save = point.x
          @@y_save = point.y
          @@cut_depth_save = cut_depth
        }
        returnPoint = Geom::Point3d.new(point.x, point.y, cut_depth)
      rescue
        UI.messagebox("Exception in millFlatEdge "+$!)
      end
      return returnPoint
    end

    def GcodeUtil.enter_file_dialog(model=Sketchup.active_model)
      output_directory_name = PhlatScript.cncFileDir
      output_filename = PhlatScript.cncFileName
      status = false
      result = UI.savepanel(PhlatScript.getString("Save CNC File"), output_directory_name, output_filename)
      if(result != nil)
        PhlatScript.cncFile=result
        status = true
      end
      status
    end

  end

end